/*
 *  find.js
 *
 *  Copyright: (c) 2018 FileMaker, Inc. All rights reserved.
 *
 *  FILEMAKER CONFIDENTIAL
 *  This file includes unpublished proprietary source code of FileMaker,
 *  Inc. The copyright notice above does not evidence any actual or
 *  intended publication of such source code.
 */
'use strict';
var errorHandler = require('../routes/error');
var util = require('../routes/util');

/**
 * @api {POST} /find/:solution/:layout/  Find Record
 * @apiDescription find records for given filter set.
 * @apiName Find
 * @apiGroup Find
 *
 * @apiUse AccessTokenHeader
 *
 * @apiParam {json} query	fields and find criterias. <br>
 * {query: [
 *    { <fieldName >: <fieldValue>, "omit" : true to set (optional) },
 *    ...
 *    { <fieldName >: <fieldValue>, "omit" : true to set (optional) }
 * ]}
 * <fieldName> is a string with the name of field as defined in table
 * <fieldValue> is a string with the find criteria as entered into Pro (NOTE: " \ and / characters must be escaped using \" \\ and \/ in strings per JSON spec)
 * When performing a find, query property in payload defines the find criteria. The value of the query property is an array of objects that each represent a find criteria. Each object in the array corresponds to an individual find request. To mark a find request as an omit request, add the "omit" property with a value of true (JSON boolean value, so no quotes).
 * All FileMaker Pro operators are planned to be supported. In other words, whatever a user can type into a field in a find request should be allowed assuming that value can be encoded in a JSON string value.
 * Operators: =, ==, !, <, ≤ or <=, >, ≥ or >=, ..., //, ?, @, #, *, \, "", ~
 * Find requests are applied in the order specified in the array<br>
 *
 * Example: Find records with Company field ends with "Hospital" and Group is "Nurse": <br>
 * {query: [ { "Company" : "*Hospital", "Group": "=Nurse"}]}<br>
 * Find records with "Group" is "Nurse" or "Group" is "Surgeon": <br>
 * {query: [ { "Group": "=Nurse"}, {"Group": "=Surgeon"}]} <br>
 *
 * @apiParam {json} sort	Sort Field Specification:  <br>
 * fieldName : <field name>  (field name to sort on)<br>
 * sortOrder : <ascend, descend, or value list name><br>
 * Possible errors: Not Valid Value (507) (usually JSON parse error), Invalid Sort (404) (incorrectly specified sort), Value List Missing (108) (can't find named value list)
 * Note: sort is only applied if record set size is greater than 1 <br>
 * Precedence is inferred from order of sort field specifications in the array<br>
 * NOTE: Sort by summary field is not supported.<br>
 * Example: <br>
 * "query": { ..., "sort" : [ { "fieldName": "LastName", "sortOrder": "ascend" }, { "fieldName": "FirstName", "sortOrder": "ascend" } ] }
 *
 * @apiParamExample {json} Request-Body-Example:
 *    "query": [{"Group": "=Surgeon", "Work State" : "CA"}, {"Group": "=Surgeon", "Work State" : "NY"}],
 			"sort": [ { "fieldName": "LastName", "sortOrder": "ascend" }, { "fieldName": "FirstName", "sortOrder": "ascend" } ]
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of row objects
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "data": [record1, record2, ...]
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /find/:solution/:layout
 */
module.exports.find = function(req, res, next){
	var ipStr = util.parseClientIp(req);	

	var params = {
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	};

	var result = util.setAuthentication(req, res, params, false);
	if(result){
		var acceptedParams = ['query','sort','portal','offset','limit','script','script.param','script.prerequest','script.prerequest.param','script.presort','script.presort.param','layout.response','dateformats'];

		util.parseParams(acceptedParams, req.body, params);

		if(params['query']){
			params.query = (typeof params['query'] === "object") ? JSON.stringify(params['query']) : params['query'].toString();
		}else{
			return errorHandler.handleError('BadRequest', req, res, "Query parameter is missing.", "10");
		}
		if(params['sort']){
			params.sort = (typeof params['sort'] === "object") ? JSON.stringify(params['sort']) : params['sort'].toString();
		}
		if(params['portal']){
			params.portal = (typeof params['portal'] === "object") ? JSON.stringify(params['portal']) : params['portal'].toString();
		}
		if(params.hasOwnProperty('offset')){
			if (isNaN(params['offset']) || params['offset']<=0)  {
				return errorHandler.handleError('BadRequest', req, res, 'The offset must be an integer with value greater than 0.', "960")
			} else if (typeof params['offset'] != 'string') {
				params['offset'] = JSON.stringify(params['offset']);
			}
		}
		if(params.hasOwnProperty('limit')){
			if (isNaN(params['limit']) || params['limit']<=0)  {
				return errorHandler.handleError('BadRequest', req, res, 'The limit must be an integer with value greater than 0.', "960")
			} else if (typeof params['limit'] != 'string') {
				params['limit'] = JSON.stringify(params['limit']);
			}
		}
		if(params.hasOwnProperty('dateformats')){
			if (isNaN(params['dateformats']) || params['dateformats']<0 || params['dateformats']>2)  {
				return errorHandler.handleError('BadRequest', req, res, 'The dateformats must be an integer with value 0, 1 or 2.', "960")
			} else if (typeof params['dateformats'] != 'string') {
				params['dateformats'] = JSON.stringify(params['dateformats']);
			}
		}
		if (params.hasOwnProperty('script.param') && params['script.param'] && !params.hasOwnProperty('script')) {
			return errorHandler.handleError('BadRequest', req, res, "script.param without script","960");				
		} else if (params.hasOwnProperty('script.prerequest.param') && params['script.prerequest.param'] && !params.hasOwnProperty('script.prerequest')) {
			return errorHandler.handleError('BadRequest', req, res, "script.prerequest.param without script.prerequest","960");				
		} else if (params.hasOwnProperty('script.presort.param') && params['script.presort.param'] && !params.hasOwnProperty('script.presort')) {
			return errorHandler.handleError('BadRequest', req, res, "script.presort.param without script.presort","960");				
		}

		for (var propName in req.body) {
			if (propName.indexOf("offset.") === 0 || (propName.indexOf("limit.") === 0)){
				if (isNaN(req.body[propName]) || req.body[propName]<=0)  {
					return errorHandler.handleError('BadRequest', req, res, 'The '+ propName +' must be an integer with value greater than 0.',"960");
				}
				params[propName.toLowerCase()] = (typeof req.body[propName] != 'string') ? JSON.stringify(req.body[propName]) : req.body[propName];
				delete req.body[propName];
			}
		}

		var remainingProps = Object.keys(req.body);
		if (remainingProps.length > 0) {
			return errorHandler.handleError("BadRequest", req, res, "Unknown parameter(s): "+remainingProps.join(','),"960");		
		}

		try{
			util.thrift_client.findRecords(params,
			function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};
